(function(_window, _document, _calculateHeight, _log, _utils){
    'use strict';
    var DEBUG = false,
        UID = 0,
        targetElements = [
            ['div[style]'],
            ['table[width]', 'table[style]']
        ],
        EFFECTIVE_THRESHOLD = 0.75,
        ZOOM_THRESHOLD = 0.2,
        KEEP_DEFAULT_WIDTH = -1,
        targetAttributes = [
            'width', 'style.width', 'style.minWidth', 'style.border', 'style.zoom'
        ],
        processingHistory = [];

    /**
     * Reset the change logs
     */
    function resetProcessingHistory() {
        processingHistory = [];
    }

    /**
     * Get a deep value from object
     * @param  {object]} obj - Source object
     * @param  {string|string[]} path - Path to get value
     * @param  {*} defaultValue - Default value, if the actual value is not valid
     * @return {*} The value stored in the path.  Otherwise, defaultValue or undefined if not specified
     */
    function getValue (obj, path, defaultValue) {
        var parts = typeof path === 'string' ? path.split('.') : path,
            part,
            i,
            len;

        for (i = 0, len = parts.length; i < len; i++) {
            part = parts[i];

            if (obj !== null && typeof obj === 'object' && part in obj) {
                obj = obj[part];
            } else {
                return defaultValue;
            }
        }

        return obj;
    }

    /**
     * Set a deep value to object
     * @param  {object]} obj - Destination object
     * @param  {string|string[]} path - Path to set value
     * @param  {*} value - The value to be stored in the path.
     */
    function setValue (obj, path, value) {
        var e;

        if (typeof path === 'string') {
            path = path.split('.');
        }

        if (path.length > 1) {
            e = path.shift();

            if (typeof obj[e] === 'object') {
                setValue(
                    obj[e],
                    path,
                    value
                );
            }
        } else {
            obj[path[0]] = value;
        }
    }

    /**
     * Try to fix/remove all elements width attributes in "targetElements", which defined at the beginning.
     * The algorithm is that if the element has an inline width, which is greater than the current window.innerWidth,
     * we should fix it, since it might make the content wide
     */
    function _preProcessBodyContent() {
        var minTargetWidth = window.document.body.clientWidth
        var originalWindowSize = window.innerWidth,
            goalWidth = minTargetWidth,
            startTime,
            dirty;

        if (_log.shouldLog(_log.DEBUG)) {
            startTime = Date.now();
            _log.d('Browser detected that the optimized width should be ' + goalWidth);
        }

        dirty = normalizeBody(goalWidth);

        if (_log.shouldLog(_log.DEBUG)) {
            _log.d('Processing HTML takes ' + (Date.now() - startTime) + ' milliseconds!');
        }

        if (window.innerWidth / originalWindowSize < 0.75 && dirty) {
            _log.d('Showing "show original" button');
            ConversationInterface.notifyTransformed(true);
            _log.d('Window innerWidth' + _window.innerWidth);
            _log.d('Window originalWindowSize' + originalWindowSize);
        } else {
            ConversationInterface.notifyTransformed(false);
        }

        ConversationInterface.scaleToFit(Math.max(goalWidth, _window.innerWidth));
    }


    /**
     * Util function to set a pixel value to an element
     * @param {object} el    - The element to be changed
     * @param {string} attr  - The attribute of the element to be changed
     * @param {number} value - The value to be set to the element
     */
    function setPixelStyle(el, attr, value) {
        el.style[attr] = value + 'px';
    }

    /**
     * Normalize body to match the given goalWidth.
     * We will remove/fix all inline width attributes/styles.
     * If the width of each element is still wider than the ideal width, we will scale it.
     * @param  {object} elements - The elements need to match the goalWidth
     * @param  {number} goalWidth - The ideal width of all elements
     */
    function normalizeBody(goalWidth) {
        var zoomChanged = false,
            contentChanged = false,
            widthChanged = false,
            oldWidth,
            zoom,
            element = _document.body;

        // For Idempotence of current function
        revertChanges();

        saveHistory(element);

        oldWidth = getValue(element, 'style.width', '');

        setPixelStyle(element, 'width', goalWidth);

        _log.d('Set body to goalWidth: ' + goalWidth);

        widthChanged = true;

        contentChanged = normalizeElementWidth(element, goalWidth);

        if (Math.abs(_window.innerWidth - goalWidth) > 1) {
            // If window width doesn't follow body width, which means there are some other content
            // inside body which stretch the window
            // The hard coded width won't help, we need reset it to original value.

            widthChanged = false;
            element.style.width = oldWidth;

            if (_log.shouldLog(_log.DEBUG)) {
                _log.d('Reset body to oldWidth: ' + oldWidth);
            }
        }

        if (_log.shouldLog(_log.DEBUG)) {
            _log.d('window innerWidth: ' + _window.innerWidth);
            _log.d('Element scrollWidth: ' + element.scrollWidth);
        }

        // We need to compare body(element.scrollWidth) and window.innerWidth
        // to know whether we have some offscreen content
        // If we do, we need to reset the body.width and zoom out the body element to display all content on screen
        // Note that we allow 1px different because different system may have their own way to round decimal
        if (element.scrollWidth - _window.innerWidth > 1) {
            // Zoom out the element to display all of them in screen

            zoom = _window.innerWidth / element.scrollWidth;

            // The logic below is trying to zoom out to let user to see all email body,
            // However, if we zoom out too much, user will see all content but the font will be too tiny
            // So we only allow to zoom out at most certain amount
            // Otherwise, user will just see partial page but the font size will be at least acceptable.
            if (zoom > ZOOM_THRESHOLD) {
                element.style.zoom = zoom;
                _log.d('Set zoom to: ' + element.style.zoom);
                zoomChanged = true;
            }
        }

        if (_log.shouldLog(_log.DEBUG)) {
            _log.d('Normalize body done');
            _log.d('Window innerWidth: ' + _window.innerWidth);
        }

        if (DEBUG && element.style.zoom !== 1) {
            element.style.border = "2px dotted red";
        }

        return contentChanged || zoomChanged || widthChanged;
    }


    /**
     * Try to normalize the width of the matched nodes inside of a given element to match the goalWidth
     * If after modification, the width of the root is not changed significantly, we would revert the changes.
     * @param  {object} element - The root element
     * @param  {number} goalWidth - The ideal width of the element
     */
    function normalizeElementWidth(element, goalWidth) {
        var i,
            dirty = false,
            touched = false,
            finished = false,
            originalWidth = element.scrollWidth;

        for (i = 0; i < targetElements.length; i++) {
            touched = normalizeTargetElements(element.querySelectorAll(targetElements[i].join(',')), goalWidth);
            dirty = dirty || touched;

            if (element.scrollWidth <= goalWidth) {
                finished = true;
                break;
            }
        }

        if (!finished &&
                dirty &&
                ((element.scrollWidth - goalWidth) / (originalWidth - goalWidth) > EFFECTIVE_THRESHOLD)) {
            // Sometimes, even though we modified all the suspicious content,
            // the width still doesn't change significantly at all
            // This case is dangerous in sometimes, because this kind of emails may have a very complicated structure
            // and layout structure might be destroyed already at this point, we need to do a revert.
            _log.d('Revert changes');
            revertChanges();
            dirty = false;
        }

        return dirty;
    }

    /**
     * Purge the inline attributes and styles
     * @param  {object} elements  - Elements to be purged
     * @param  {number} goalWidth - The ideal width of all elements
     * @return {boolean} Whether these elements get changed or not.
     */
    function normalizeTargetElements(elements, goalWidth) {
        var i,
            len = elements.length,
            element,
            touched = false,
            dirty = false;

        for (i = 0; i < len; i++) {
            element = elements[i];

            if (element.tagName === "TABLE") {
                touched = normalizeTargetTableElement(element, goalWidth);
            } else {
                touched = normalizeTargetElement(element, goalWidth);
            }

            dirty = dirty || touched;
        }

        return dirty;
    }

    /**
     * Purge a single element.
     * If it has a inline style that is greater than goalWidth,
     * we would save the old styles and remove the suspicious styles
     * @param  {object} element   - Element to be purged
     * @param  {number} goalWidth - The ideal width of all element
     * @return {boolean} Whether these elements get changed or not.
     */
    function normalizeTargetElement(element, goalWidth) {

        // Inline width
        // TODO: need to know which one is currently in use
        var widthStr = element.style.width || element.style.minWidth || element.width,
            width = widthStr ? widthStr.replace('px', '') : '';

        // The line below looks hacky, because it is comparing between a string and a number,
        // However, this is a simple and efficient way to handle the possibility of '100', '100em'
        // and '100%', instead of doing expensive 'parseInt' and check the existence of '%' or 'em'
        if (width > goalWidth) {
            saveHistory(element);
            fixAttribute(element, width);

            return true;
        }

        return false;
    }

    /**
     * Purge a single table element.
     * If it has a inline style that is greater than goalWidth,
     * we would save the old styles and remove the suspicious styles
     * @param  {object} element   - Element to be purged
     * @param  {number} goalWidth - The ideal width of all element
     * @return {boolean} Whether these elements get changed or not.
     */
    function normalizeTargetTableElement(element, goalWidth) {
        var dirty = false,
            tdElements;

        if (element.style.tableLayout !== 'fixed') {
            // Fixed table layout require child element size
            // It's might be really dangers to mess up their width
            // Only try to "fix" tds inside of non-fixed table

            tdElements = getTdElements(element);

            dirty = normalizeTargetElements(tdElements, goalWidth);
        }

        return normalizeTargetElement(element, goalWidth) || dirty;
    }

    /**
     * Get all tds belong to current table element
     */
    function getTdElements(element) {
        var elements = [],
            oldElId,
            randomId;

        try {
            elements = element.querySelectorAll(':scope > tbody > tr > td');
        } catch(e) {}

        if (!elements || elements.length === 0) { // For Webview doesn't support ":scope"
            oldElId = element.id;
            randomId = 'ymail_unique_id_' + UID++;
            element.id = randomId;
            elements = element.querySelectorAll('#' + randomId + ' > tbody > tr > td');
            element.id = oldElId;
        }

        return elements;
    }

    /**
     * Set inline width to be 'auto', style.width to be 100% , remove style.minWidth
     * and set the maxWidth to be the width that passed in
     *
     * @param  {object} element   - Element to be purged
     * @param  {number} width - The original width of current element
     */
    function fixAttribute(element, width) {
        element.width = 'auto';
        element.style.width = 'auto';
        element.style.minWidth = '';
        if (DEBUG) {
            element.style.border = "2px dotted red";
        }
        setPixelStyle(element, 'maxWidth', width);
    }

    /**
     * Save the element original width attributes and width style
     * @param  {object} el - The source element
     */
    function saveHistory(el) {
        var i,
            targetAttribute,
            len = targetAttributes.length,
            data = [];

        for (i = 0; i < len; i++) {
            targetAttribute = targetAttributes[i];
            data.push({
                attr: targetAttribute,
                value: getValue(el, targetAttribute, '')
            });
        }

        processingHistory.push({
            el: el,
            data: data
        });
    }

    /**
     * Revert all the actions that saved inside of the change history
     */
    function revertChanges() {
        var i,
            j,
            item,
            len2,
            el,
            data,
            len;

        for (i = 0, len = processingHistory.length; i < len; i++) {
            item = processingHistory[i];
            el = item.el;
            data = item.data;

            for (j = 0, len2 = data.length; j < len2; j++) {
                setValue(el, data[j].attr, data[j].value);
            }
        }

        resetProcessingHistory();
    }

    /**
     * Function to precess email body when all elements loaded.
     * Note: there's no need for us the expose current function since it will never be called externally
     * However, we might need to pass in the actual pixel value from java, if "window.innerWidth" doesn't work well
     * in older android version. So leave this function exposed for now.
     */
    function init () {

        _document.addEventListener('DOMContentLoaded', function() {
            var resizeWithDebounce,
                startTime = 0;

            if (_log.shouldLog(_log.DEBUG)) {
                startTime = Date.now();
                _log.d('DOMContentLoaded Loaded');
                _log.d('Window innerWidth', _window.innerWidth);
            }

            function onResizeCallback() {

                var minTargetWidth = _window.document.body.clientWidth
                if ( minTargetWidth != 0 && (_window.innerWidth >= minTargetWidth || _window.innerWidth + 1 == minTargetWidth)) {
                    var divNodes = _window.document.body.children
                    if (divNodes.length > 0 && divNodes[0] != null) {
                        _log.d('moving content offscreen to avoid flickering when determining height and zoom scale');
                        divNodes[0].style.left = -999999
                    }

                    if (_log.shouldLog(_log.DEBUG)) {
                        _log.d('Starting preProcessBodyContent: ' + ( Date.now() - startTime ) +
                                ' ms after dom content loaded');
                    }

                    _window.removeEventListener('resize', resizeWithDebounce);
                    _preProcessBodyContent();
                    _calculateHeight();

                    if (divNodes.length > 0 && divNodes[0] != null) {
                        _log.d('bring content back to screen');
                        divNodes[0].style.left = 0
                    }

                } else {
                    ConversationInterface.scaleToFit(KEEP_DEFAULT_WIDTH);
                }
            }

            resizeWithDebounce = _utils.debounce(onResizeCallback, 100, false);

            _window.addEventListener('resize', resizeWithDebounce);

            // Put at lease one cb into Debounce queue
            onResizeCallback();

            var MAX_TEXT_CONTENT_LENGTH = 256;
            _document.body.addEventListener("click", function(e) {
                var target = e.target,
                    anchor = target.closest("a");
                var anchorLink = target.closest('a[href^="#"]');

                if (anchorLink) {
                    e.preventDefault();
                    const linkId = (anchorLink.getAttribute("href") || "").substring(1);
                    if (linkId.length == 0) return;
                    var innerText = anchorLink.innerText || '';

                    const YIV_CONTENT_ID_SELECTOR = 'body > div[id]:first-child';
                    const yivContentElement = _document.querySelector(YIV_CONTENT_ID_SELECTOR);
                    const yivContentId = yivContentElement && yivContentElement.id;
                    const destinationSelector = `[id="${linkId}"],[id="${yivContentId}${linkId}"],a[name="${yivContentId}${linkId}"],a[name="${linkId}"]`;
                    const destinationElement = _document.querySelector(destinationSelector);
                    const topPosition = Math.round(destinationElement.getBoundingClientRect().y);
                    const tappedElementPosition = Math.round(target.getBoundingClientRect().y);
                    const webHeight = topPosition - tappedElementPosition;
                    const webWidth = _window.innerWidth;

                     ConversationInterface.handleAnchorLinkTap(anchorLink.getAttribute("href"), innerText, webWidth, webHeight);
                } else if (anchor) {
                    var textContent = anchor.innerText || '';

                    // Check alt text in case an image was clicked
                    for (let i = 0; i < anchor.childElementCount; i++) {
                        if (anchor.children[i].tagName === 'IMG' && anchor.children[i].alt) {
                            textContent = anchor.children[i].alt;
                            break;
                        }
                    }
                    textContent = textContent.trim().slice(0, MAX_TEXT_CONTENT_LENGTH);
                    if (anchor.onclick == null) {
                        ConversationInterface.handleLinkClick(anchor.getAttribute("href"), textContent);
                    }
                }
             });
        });
    }

    function _hideBody() {
        _document.body.style.opacity = 0;
    }

    function _showBody() {
        _document.body.style.opacity = 1;
    }

    _window.showOriginalEmail = function(showOrigin) {
        _hideBody();

        if (showOrigin) {
            revertChanges();
        } else {
            normalizeBody(window.document.body.clientWidth);
        }

        _showBody();
        _window.calculateHeight();
    };

    _document.addEventListener("DOMContentLoaded", function() {
        var elements = _document.getElementsByTagName('img'),
            element,
            i;

        for (i = 0; i < elements.length; i++) {
            element = elements[i];

            //if ios/desktop sends cloud attachment they arrive with different markup. For them, we want to avoid setting max-width for the image and leave with whatever they come with
            if ((!element.className || (element.className.indexOf('card-thumbnail-image') == -1 && element.className.indexOf('yahoo-ignore-inline-image') == -1)) &&
                !element.style.width && !element.style.minWidth && !element.getAttribute('width')) {
                element.style.maxWidth = '100%';
            }
        }
    });

    // Kick off the preProcessBodyContent logic to process
    init();

    window.formatMessage = _preProcessBodyContent;

})(window, document, window.calculateHeight, window.log, window.utils);
